(function () {
  const BUTTON_ID = "extractik-float-btn";
  const PANEL_ID = "extractik-export-panel";
  const OVERLAY_ID = "extractik-export-overlay";

  let pollTimer = null;

  function toSafeFileName(value) {
    return (value || "tiktok_profile")
      .replace(/[^\w.-]+/g, "_")
      .replace(/^_+|_+$/g, "")
      .toLowerCase();
  }

  function parseUsername(pathname) {
    const match = pathname.match(/^\/@([^/?#]+)/);
    return match ? match[1] : "";
  }

  function isProfilePath(pathname) {
    return /^\/@[^/?#]+\/?$/.test(pathname);
  }

  function formatNumber(num) {
    if (typeof num !== "number" || Number.isNaN(num)) return "0";
    return num.toLocaleString();
  }

  function toNumber(value) {
    if (typeof value === "number") return value;
    if (typeof value === "string") {
      const clean = value.replace(/[^\d.-]/g, "");
      const parsed = parseInt(clean, 10);
      return Number.isNaN(parsed) ? 0 : parsed;
    }
    return 0;
  }

  function escapeHtml(value) {
    return String(value ?? "")
      .replace(/&/g, "&amp;")
      .replace(/</g, "&lt;")
      .replace(/>/g, "&gt;")
      .replace(/"/g, "&quot;")
      .replace(/'/g, "&#39;");
  }

  function getInitials(name) {
    const text = String(name || "").trim();
    if (!text) return "U";
    const parts = text.split(/\s+/).slice(0, 2);
    return parts.map((part) => part[0]?.toUpperCase() || "").join("") || "U";
  }

  function renderAvatar(url, label, sizePx) {
    const size = Number(sizePx) || 36;
    const safeLabel = escapeHtml(label || "User");
    if (url) {
      return `<img src="${escapeHtml(url)}" alt="${safeLabel}" style="width:${size}px;height:${size}px;border-radius:999px;object-fit:cover;border:1px solid #33476a;" />`;
    }
    return `<div style="width:${size}px;height:${size}px;border-radius:999px;display:flex;align-items:center;justify-content:center;background:#1a2234;border:1px solid #33476a;color:#d6e0f7;font-size:12px;font-weight:700;">${getInitials(label)}</div>`;
  }

  function removeButton() {
    const existing = document.getElementById(BUTTON_ID);
    if (existing) existing.remove();
  }

  function stopPolling() {
    if (pollTimer) {
      clearTimeout(pollTimer);
      pollTimer = null;
    }
  }

  function removePanel() {
    stopPolling();
    const panel = document.getElementById(PANEL_ID);
    if (panel) panel.remove();
    const overlay = document.getElementById(OVERLAY_ID);
    if (overlay) overlay.remove();
  }

  function downloadBlob(content, type, filename) {
    const blob = new Blob([content], { type });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    a.remove();
    URL.revokeObjectURL(url);
  }

  function toCsvRows(exportData) {
    const posts = Array.isArray(exportData?.posts) ? exportData.posts : [];
    return posts.map((post) => ({
      "Post ID": post?.id || "",
      "Title": post?.title || "",
      "Author": post?.author?.nickname || post?.author?.uniqueId || "",
      "Username": post?.author?.uniqueId || "",
      "Duration (ms)": post?.duration || "",
      "Views": post?.play_count || "",
      "Likes": post?.digg_count || "",
      "Comments": post?.comment_count || "",
      "Shares": post?.share_count || "",
      "Downloads": post?.download_count || "",
      "Collects": post?.collect_count || "",
      "Music Title": post?.music?.title || "",
      "Music Author": post?.music?.author_name || "",
      "Download URL": post?.dowload_url || "",
      "No Watermark URL": post?.dowload_no_watermark_url || "",
    }));
  }

  function toCsv(data) {
    const rows = toCsvRows(data);
    if (!rows.length) {
      return "\"No data\"\n";
    }

    const headers = Object.keys(rows[0]);
    const lines = [headers];

    rows.forEach((row) => {
      lines.push(headers.map((header) => row[header]));
    });

    return lines
      .map((line) =>
        line
          .map((cell) => `"${String(cell ?? "").replace(/"/g, '""')}"`)
          .join(",")
      )
      .join("\n");
  }

  function sendRuntimeMessage(message) {
    return new Promise((resolve, reject) => {
      chrome.runtime.sendMessage(message, (response) => {
        if (chrome.runtime.lastError) {
          reject(new Error(chrome.runtime.lastError.message));
          return;
        }
        resolve(response);
      });
    });
  }

  async function apiRequest(path, options) {
    const response = await sendRuntimeMessage({
      type: "EXTRACTIK_API_REQUEST",
      path,
      method: options?.method || "GET",
      query: options?.query,
      body: options?.body,
    });

    return response || { ok: false, status: 0, data: null };
  }

  function normalizeProfile(data, username) {
    let responseData = data;
    if (data?.user) responseData = data.user;
    else if (data?.data) responseData = data.data;

    const user = responseData?.user || responseData || {};
    const stats = responseData?.stats || {};
    const statsV2 = responseData?.statsV2 || {};

    return {
      username: user.uniqueId || user.username || username,
      nickname: user.nickname || user.username || username,
      secUid: user.secUid || user.sec_uid || responseData?.secUid || responseData?.sec_uid || "",
      userId: user.id || user.userId || responseData?.id || responseData?.userId || "",
      avatar: user.avatarThumb || user.avatarMedium || user.avatarLarger || user.avatar || "",
      isVerified: Boolean(user.verified),
      followers:
        typeof stats.followerCount === "number"
          ? stats.followerCount
          : toNumber(statsV2.followerCount),
      following:
        typeof stats.followingCount === "number"
          ? stats.followingCount
          : toNumber(statsV2.followingCount),
      likes:
        typeof stats.heartCount === "number"
          ? stats.heartCount
          : toNumber(statsV2.heartCount),
      videos:
        typeof stats.videoCount === "number"
          ? stats.videoCount
          : toNumber(statsV2.videoCount),
    };
  }

  function openLogin() {
    chrome.runtime.sendMessage({ type: "OPEN_EXTRACTIK_LOGIN" });
  }

  function createPanelShell() {
    removePanel();

    const overlay = document.createElement("div");
    overlay.id = OVERLAY_ID;
    Object.assign(overlay.style, {
      position: "fixed",
      inset: "0",
      background: "rgba(0,0,0,0.45)",
      zIndex: "999998",
    });

    const panel = document.createElement("div");
    panel.id = PANEL_ID;
    Object.assign(panel.style, {
      position: "fixed",
      right: "16px",
      bottom: "72px",
      width: "min(390px, calc(100vw - 24px))",
      background: "#0f1624",
      color: "#f5f7ff",
      border: "1px solid #2d3c5a",
      borderRadius: "14px",
      padding: "14px",
      zIndex: "999999",
      fontFamily: "system-ui, -apple-system, Segoe UI, sans-serif",
      boxShadow: "0 16px 36px rgba(0, 0, 0, 0.35)",
    });

    function closePanel() {
      removePanel();
    }

    overlay.addEventListener("click", closePanel);

    document.body.appendChild(overlay);
    document.body.appendChild(panel);

    return panel;
  }

  function getErrorMessage(response, fallback) {
    if (response?.data && typeof response.data === "object") {
      if (response.data.message) return response.data.message;
      if (response.data.error) return response.data.error;
    }
    if (typeof response?.data === "string" && response.data.trim()) {
      return response.data;
    }
    return fallback;
  }

  function renderHeader(username) {
    return `
      <div style="display:flex; align-items:center; justify-content:space-between; gap:8px;">
        <strong style="font-size:14px;">Export @${username}</strong>
        <button id="extractik-close-panel" style="border:none;background:transparent;color:#aeb9d6;font-size:18px;cursor:pointer;line-height:1;">×</button>
      </div>
    `;
  }

  function renderSummaryCards(options) {
    const {
      loggedInName,
      loggedInEmail,
      loggedInAvatar,
      credits,
      profile,
      totalPosts,
      maxExportablePosts,
    } = options;

    const creditsText = credits === null ? "-" : formatNumber(credits);
    const safeLoggedInName = escapeHtml(loggedInName || "Extractik user");
    const safeLoggedInEmail = escapeHtml(loggedInEmail || "");
    const safeNickname = escapeHtml(profile.nickname || profile.username || "TikTok profile");
    const safeUsername = escapeHtml(`@${String(profile.username || "").replace(/^@/, "")}`);

    return `
      <div style="margin-top:10px;display:grid;gap:10px;">
        <div style="display:flex;align-items:center;justify-content:space-between;gap:10px;padding:10px;border:1px solid #2d3c5a;border-radius:12px;background:#101a2b;">
          <div style="display:flex;align-items:center;gap:10px;min-width:0;">
            ${renderAvatar(loggedInAvatar, loggedInName, 36)}
            <div style="min-width:0;">
              <p style="margin:0;color:#f5f7ff;font-size:12px;font-weight:700;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;">${safeLoggedInName}</p>
              <p style="margin:2px 0 0;color:#8fa1c7;font-size:11px;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;">${safeLoggedInEmail || "Logged in to Extractik"}</p>
            </div>
          </div>
          <div style="text-align:right;">
            <p style="margin:0;color:#8fa1c7;font-size:10px;text-transform:uppercase;letter-spacing:.08em;">Credits</p>
            <p style="margin:2px 0 0;color:#00f2ea;font-size:16px;font-weight:800;">${creditsText}</p>
          </div>
        </div>

        <div style="padding:10px;border:1px solid #2d3c5a;border-radius:12px;background:#101a2b;">
          <div style="display:flex;align-items:center;gap:10px;">
            ${renderAvatar(profile.avatar, profile.nickname || profile.username, 40)}
            <div style="min-width:0;">
              <p style="margin:0;color:#f5f7ff;font-size:13px;font-weight:700;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;">${safeNickname}${profile.isVerified ? " ✓" : ""}</p>
              <p style="margin:2px 0 0;color:#8fa1c7;font-size:11px;">${safeUsername}</p>
            </div>
          </div>
          <div style="margin-top:10px;display:grid;grid-template-columns:1fr 1fr 1fr;gap:8px;">
            <div style="padding:8px;border-radius:10px;background:#0b1322;border:1px solid #27395b;">
              <p style="margin:0;color:#8fa1c7;font-size:10px;">Followers</p>
              <p style="margin:3px 0 0;color:#f5f7ff;font-size:12px;font-weight:700;">${formatNumber(profile.followers)}</p>
            </div>
            <div style="padding:8px;border-radius:10px;background:#0b1322;border:1px solid #27395b;">
              <p style="margin:0;color:#8fa1c7;font-size:10px;">Following</p>
              <p style="margin:3px 0 0;color:#f5f7ff;font-size:12px;font-weight:700;">${formatNumber(profile.following)}</p>
            </div>
            <div style="padding:8px;border-radius:10px;background:#0b1322;border:1px solid #27395b;">
              <p style="margin:0;color:#8fa1c7;font-size:10px;">Likes</p>
              <p style="margin:3px 0 0;color:#f5f7ff;font-size:12px;font-weight:700;">${formatNumber(profile.likes)}</p>
            </div>
          </div>
          <div style="margin-top:8px;display:flex;justify-content:space-between;color:#aeb9d6;font-size:11px;">
            <span>Available posts: <strong style="color:#f5f7ff;">${formatNumber(totalPosts)}</strong></span>
            <span>Max export: <strong style="color:#f5f7ff;">${formatNumber(maxExportablePosts)}</strong></span>
          </div>
        </div>
      </div>
    `;
  }

  function bindClose(panel) {
    const closeBtn = panel.querySelector("#extractik-close-panel");
    if (closeBtn) {
      closeBtn.addEventListener("click", () => removePanel());
    }
  }

  async function openExportPanel(username) {
    const panel = createPanelShell();

    panel.innerHTML = `
      ${renderHeader(username)}
      <p style="margin:8px 0 0;color:#aeb9d6;font-size:12px;line-height:1.4;">Checking your Extractik session and loading profile data...</p>
    `;
    bindClose(panel);

    const authResponse = await apiRequest("/api/auth/me");
    if (!authResponse.ok) {
      panel.innerHTML = `
        ${renderHeader(username)}
        <p style="margin:10px 0;color:#ffb4b4;font-size:12px;line-height:1.4;">You are not logged in to Extractik.</p>
        <button id="extractik-login" style="width:100%;border:none;border-radius:10px;padding:10px 12px;font-weight:700;cursor:pointer;background:linear-gradient(90deg,#00f2ea 0%,#ff0050 100%);color:#111;">Login to Extractik</button>
      `;
      bindClose(panel);
      const loginBtn = panel.querySelector("#extractik-login");
      if (loginBtn) loginBtn.addEventListener("click", openLogin);
      return;
    }

    const authData = authResponse.data && typeof authResponse.data === "object" ? authResponse.data : {};
    const loggedInName = authData.name || authData.fullName || authData.username || "Extractik user";
    const loggedInEmail = authData.email || "";
    const loggedInAvatar = authData.picture || authData.avatar || "";

    const [creditsResponse, profileResponse] = await Promise.all([
      apiRequest("/api/credits/balance"),
      apiRequest("/api/tiktok/profile", { method: "GET", query: { query: username } }),
    ]);

    if (!profileResponse.ok) {
      panel.innerHTML = `
        ${renderHeader(username)}
        <p style="margin:10px 0;color:#ffb4b4;font-size:12px;line-height:1.4;">${getErrorMessage(profileResponse, "Could not load profile from Extractik API.")}</p>
      `;
      bindClose(panel);
      return;
    }

    const profile = normalizeProfile(profileResponse.data, username);
    const totalPosts = Math.max(0, toNumber(profile.videos));
    let credits = null;

    if (creditsResponse.ok) {
      const value = creditsResponse.data;
      if (typeof value === "number") credits = value;
      else if (value && typeof value === "object") {
        credits = toNumber(value.balance ?? value.credits ?? 0);
      }
    }

    const creditLimitedMax = credits === null ? totalPosts : Math.min(totalPosts, Math.max(0, credits));
    const maxExportablePosts = Math.max(0, creditLimitedMax);
    const defaultCount = maxExportablePosts > 0 ? Math.min(100, maxExportablePosts) : 0;

    function renderReady(errorText) {
      panel.innerHTML = `
        ${renderHeader(username)}
        ${renderSummaryCards({
          loggedInName,
          loggedInEmail,
          loggedInAvatar,
          credits,
          profile,
          totalPosts,
          maxExportablePosts,
        })}
        <div style="margin-top:10px;display:grid;gap:8px;">
          <label style="font-size:12px;color:#aeb9d6;">Posts to export</label>
          <input id="extractik-post-count" type="number" min="0" max="${maxExportablePosts}" value="${defaultCount}" style="width:100%;box-sizing:border-box;border:1px solid #304160;border-radius:10px;background:#0b111d;color:#f5f7ff;padding:10px 12px;" />
          <div style="display:grid;grid-template-columns:1fr 1fr 1fr;gap:8px;">
            <button id="extractik-preset-50" style="border:none;border-radius:10px;padding:8px 10px;font-weight:600;cursor:pointer;background:#1a2234;color:#f5f7ff;">50</button>
            <button id="extractik-preset-100" style="border:none;border-radius:10px;padding:8px 10px;font-weight:600;cursor:pointer;background:#1a2234;color:#f5f7ff;">100</button>
            <button id="extractik-preset-max" style="border:none;border-radius:10px;padding:8px 10px;font-weight:600;cursor:pointer;background:#1a2234;color:#f5f7ff;">Max</button>
          </div>
          <button id="extractik-start-export" ${maxExportablePosts === 0 ? "disabled" : ""} style="border:none;border-radius:10px;padding:10px 12px;font-weight:700;cursor:${maxExportablePosts === 0 ? "not-allowed" : "pointer"};background:${maxExportablePosts === 0 ? "#2a3144" : "linear-gradient(90deg,#00f2ea 0%,#ff0050 100%)"};color:${maxExportablePosts === 0 ? "#98a2bd" : "#111"};">Start Export</button>
          ${maxExportablePosts === 0 ? "<p style=\"margin:0;color:#ffb4b4;font-size:12px;line-height:1.4;\">No posts can be exported right now. You may need more credits or this profile has no posts.</p>" : ""}
          ${errorText ? `<p style="margin:0;color:#ffb4b4;font-size:12px;line-height:1.4;">${errorText}</p>` : ""}
        </div>
      `;

      bindClose(panel);

      const input = panel.querySelector("#extractik-post-count");
      const startBtn = panel.querySelector("#extractik-start-export");
      const preset50 = panel.querySelector("#extractik-preset-50");
      const preset100 = panel.querySelector("#extractik-preset-100");
      const presetMax = panel.querySelector("#extractik-preset-max");

      if (preset50) {
        preset50.addEventListener("click", () => {
          input.value = String(Math.max(0, Math.min(50, maxExportablePosts)));
        });
      }
      if (preset100) {
        preset100.addEventListener("click", () => {
          input.value = String(Math.max(0, Math.min(100, maxExportablePosts)));
        });
      }
      if (presetMax) {
        presetMax.addEventListener("click", () => {
          input.value = String(Math.max(0, maxExportablePosts));
        });
      }

      if (startBtn) {
        startBtn.addEventListener("click", async () => {
          if (maxExportablePosts === 0) {
            renderReady("No posts can be exported right now.");
            return;
          }

          const requested = toNumber(input.value);
          const maxAllowed = maxExportablePosts;

          if (requested < 1 || requested > maxAllowed) {
            renderReady(`Post count must be between 1 and ${maxAllowed}.`);
            return;
          }

          panel.innerHTML = `
            ${renderHeader(username)}
            ${renderSummaryCards({
              loggedInName,
              loggedInEmail,
              loggedInAvatar,
              credits,
              profile,
              totalPosts,
              maxExportablePosts,
            })}
            <p style="margin:10px 0 0;color:#aeb9d6;font-size:12px;line-height:1.4;">Export queued. We are preparing ${requested} posts...</p>
          `;
          bindClose(panel);

          const queueResponse = await apiRequest("/api/tiktok/fetch", {
            method: "POST",
            body: {
              username: String(profile.username || username).replace(/^@/, ""),
              count: requested,
              ...(profile.secUid ? { secUid: profile.secUid } : {}),
              ...(profile.userId ? { id: profile.userId } : {}),
            },
          });

          if (!queueResponse.ok) {
            renderReady(getErrorMessage(queueResponse, "Failed to start export."));
            return;
          }

          const jobId = queueResponse.data?.jobId || queueResponse.data?.id;
          if (!jobId) {
            renderReady("Export started but no job ID was returned.");
            return;
          }

          const pollJob = async () => {
            const jobResponse = await apiRequest(`/api/tiktok/jobs/${jobId}`);
            if (!jobResponse.ok) {
              renderReady(getErrorMessage(jobResponse, "Failed to check export status."));
              return;
            }

            const statusValue = String(jobResponse.data?.status || "").toLowerCase();

            if (statusValue === "completed") {
              const exportData = jobResponse.data?.results || jobResponse.data?.result || jobResponse.data;
              const posts = Array.isArray(exportData?.posts) ? exportData.posts.length : 0;
              const baseName = `${toSafeFileName(username)}_export_${Date.now()}`;

              panel.innerHTML = `
                ${renderHeader(username)}
                ${renderSummaryCards({
                  loggedInName,
                  loggedInEmail,
                  loggedInAvatar,
                  credits,
                  profile,
                  totalPosts,
                  maxExportablePosts,
                })}
                <p style="margin:10px 0 0;color:#b5ffd8;font-size:12px;line-height:1.4;">Export completed. Posts returned: ${posts}</p>
                <div style="display:grid;grid-template-columns:1fr 1fr;gap:8px;">
                  <button id="extractik-download-json" style="border:none;border-radius:10px;padding:10px 12px;font-weight:700;cursor:pointer;background:#1a2234;color:#f5f7ff;">Download JSON</button>
                  <button id="extractik-download-csv" style="border:none;border-radius:10px;padding:10px 12px;font-weight:700;cursor:pointer;background:linear-gradient(90deg,#00f2ea 0%,#ff0050 100%);color:#111;">Download Excel (CSV)</button>
                </div>
                <p style="margin:8px 0 0;color:#8e9abc;font-size:11px;">This export uses the same backend data flow as Extractik platform.</p>
              `;
              bindClose(panel);

              const jsonBtn = panel.querySelector("#extractik-download-json");
              const csvBtn = panel.querySelector("#extractik-download-csv");

              if (jsonBtn) {
                jsonBtn.addEventListener("click", () => {
                  downloadBlob(
                    JSON.stringify(exportData, null, 2),
                    "application/json",
                    `${baseName}.json`
                  );
                });
              }

              if (csvBtn) {
                csvBtn.addEventListener("click", () => {
                  downloadBlob(
                    toCsv(exportData),
                    "text/csv;charset=utf-8",
                    `${baseName}.csv`
                  );
                });
              }
              return;
            }

            if (statusValue === "failed") {
              const failMessage = getErrorMessage(
                jobResponse,
                "Export failed. Credits were refunded by backend."
              );
              renderReady(
                `${failMessage} Try again with fewer posts.`
              );
              return;
            }

            panel.innerHTML = `
              ${renderHeader(username)}
              ${renderSummaryCards({
                loggedInName,
                loggedInEmail,
                loggedInAvatar,
                credits,
                profile,
                totalPosts,
                maxExportablePosts,
              })}
              <p style="margin:10px 0 0;color:#aeb9d6;font-size:12px;line-height:1.4;">Export is running... status: ${statusValue || "processing"}</p>
            `;
            bindClose(panel);

            pollTimer = setTimeout(pollJob, 3000);
          };

          pollJob();
        });
      }
    }

    renderReady("");
  }

  function createButton(username) {
    removeButton();

    const button = document.createElement("button");
    button.id = BUTTON_ID;
    button.type = "button";
    button.textContent = `Export @${username} with Extractik`;
    button.setAttribute("aria-label", "Export this profile with Extractik");

    Object.assign(button.style, {
      position: "fixed",
      right: "16px",
      bottom: "16px",
      zIndex: "999999",
      border: "none",
      borderRadius: "999px",
      padding: "12px 16px",
      background: "linear-gradient(90deg, #00f2ea 0%, #ff0050 100%)",
      color: "#111",
      fontWeight: "700",
      fontFamily: "system-ui, -apple-system, Segoe UI, sans-serif",
      fontSize: "13px",
      cursor: "pointer",
      boxShadow: "0 12px 28px rgba(0, 0, 0, 0.25)",
    });

    button.addEventListener("click", () => {
      openExportPanel(username);
    });

    document.body.appendChild(button);
  }

  function refresh() {
    const { pathname } = window.location;
    if (!isProfilePath(pathname)) {
      removeButton();
      removePanel();
      return;
    }

    const username = parseUsername(pathname);
    if (!username) {
      removeButton();
      removePanel();
      return;
    }

    createButton(username);
  }

  let lastPath = window.location.pathname;
  refresh();

  setInterval(() => {
    if (window.location.pathname !== lastPath) {
      lastPath = window.location.pathname;
      refresh();
    }
  }, 500);

  chrome.runtime.onMessage.addListener((message, _sender, sendResponse) => {
    if (message?.type === "OPEN_EXPORT_PANEL") {
      const username = parseUsername(window.location.pathname);
      if (!username) {
        sendResponse({ ok: false, error: "Not on a TikTok profile page" });
        return false;
      }

      openExportPanel(username)
        .then(() => sendResponse({ ok: true }))
        .catch((err) => sendResponse({ ok: false, error: err?.message || "Failed to open panel" }));
      return true;
    }

    return false;
  });
})();
