const DEFAULT_APP_BASE_URL = "https://www.extractik.com";
const LEGACY_APP_BASE_URLS = [
  "https://extractik.com",
  "https://extractik.mmardi.com",
  "https://tiktok-scrapper-extractik-frontend.vercel.app",
];
const KNOWN_APP_BASE_URLS = [DEFAULT_APP_BASE_URL, ...LEGACY_APP_BASE_URLS];

function normalizeBaseUrl(url) {
  return (url || "").trim().replace(/\/$/, "");
}

async function getAppBaseUrl() {
  const stored = await chrome.storage.sync.get(["appBaseUrl"]);
  const normalized = normalizeBaseUrl(stored.appBaseUrl || DEFAULT_APP_BASE_URL);
  if (LEGACY_APP_BASE_URLS.includes(normalized)) {
    return DEFAULT_APP_BASE_URL;
  }
  return normalized;
}

function ensureAppBaseUrl() {
  chrome.storage.sync.get(["appBaseUrl"], (stored) => {
    const normalized = normalizeBaseUrl(stored.appBaseUrl);
    if (!normalized || LEGACY_APP_BASE_URLS.includes(normalized)) {
      chrome.storage.sync.set({ appBaseUrl: DEFAULT_APP_BASE_URL });
    }
  });
}

function cleanUsername(username) {
  const value = (username || "").trim();
  return value.replace(/^@/, "");
}

async function getCandidateBaseUrls() {
  const stored = await getAppBaseUrl();
  const seen = new Set();
  const candidates = [];

  [stored, ...KNOWN_APP_BASE_URLS].forEach((baseUrl) => {
    const normalized = normalizeBaseUrl(baseUrl);
    if (!normalized || seen.has(normalized)) return;
    seen.add(normalized);
    candidates.push(normalized);
  });

  return candidates;
}

async function getAccessTokenForBaseUrl(appBaseUrl) {
  try {
    const cookie = await chrome.cookies.get({
      url: appBaseUrl,
      name: "access_token",
    });
    return cookie?.value || "";
  } catch {
    return "";
  }
}

async function resolveBaseUrlAndToken() {
  const candidates = await getCandidateBaseUrls();

  for (const candidate of candidates) {
    const token = await getAccessTokenForBaseUrl(candidate);
    if (!token) continue;

    const current = await getAppBaseUrl();
    if (candidate !== current) {
      chrome.storage.sync.set({ appBaseUrl: candidate });
    }

    return { appBaseUrl: candidate, token };
  }

  const fallbackBaseUrl = candidates[0] || DEFAULT_APP_BASE_URL;
  const fallbackToken = await getAccessTokenForBaseUrl(fallbackBaseUrl);
  return { appBaseUrl: fallbackBaseUrl, token: fallbackToken };
}

function buildApiUrl(appBaseUrl, path, query) {
  const normalizedPath = path.startsWith("/") ? path : `/${path}`;
  const url = new URL(`${appBaseUrl}${normalizedPath}`);

  if (query && typeof query === "object") {
    Object.entries(query).forEach(([key, value]) => {
      if (value === undefined || value === null || value === "") return;
      url.searchParams.set(key, String(value));
    });
  }

  return url.toString();
}

async function requestApi(message) {
  const { appBaseUrl, token } = await resolveBaseUrlAndToken();
  const method = (message.method || "GET").toUpperCase();
  const url = buildApiUrl(appBaseUrl, message.path || "/", message.query);

  const headers = {};
  if (method !== "GET") {
    headers["Content-Type"] = "application/json";
  }
  if (token) {
    headers.Authorization = `Bearer ${token}`;
  }

  const response = await fetch(url, {
    method,
    headers,
    credentials: "include",
    body: method === "GET" || message.body === undefined ? undefined : JSON.stringify(message.body),
  });

  const rawText = await response.text();
  let data = rawText;
  if (rawText) {
    try {
      data = JSON.parse(rawText);
    } catch {
      data = rawText;
    }
  }

  return {
    ok: response.ok,
    status: response.status,
    data,
    appBaseUrl,
  };
}

async function openExtractik(username) {
  const { appBaseUrl, token } = await resolveBaseUrlAndToken();
  const clean = cleanUsername(username);
  const nextPath = clean
    ? `/en/search?username=${encodeURIComponent(`@${clean}`)}&source=extension`
    : "/en/search";
  const signInPath = `/en/sign-in?next=${encodeURIComponent(nextPath)}`;
  const targetPath = token ? nextPath : signInPath;

  chrome.tabs.create({ url: `${appBaseUrl}${targetPath}` });
}

ensureAppBaseUrl();
chrome.runtime.onInstalled.addListener(ensureAppBaseUrl);
chrome.runtime.onStartup.addListener(ensureAppBaseUrl);

chrome.runtime.onMessage.addListener((message, _sender, sendResponse) => {
  if (message?.type === "EXTRACTIK_API_REQUEST") {
    requestApi(message)
      .then((response) => sendResponse(response))
      .catch((error) =>
        sendResponse({
          ok: false,
          status: 0,
          error: error?.message || "Request failed",
        })
      );
    return true;
  }

  if (message?.type === "OPEN_EXTRACTIK_LOGIN") {
    openExtractik("")
      .then(() => sendResponse({ ok: true }))
      .catch(() => sendResponse({ ok: false }));
    return true;
  }

  return false;
});
